// DOM Elements
const loginView = document.getElementById('loginView');
const trackerView = document.getElementById('trackerView');
const statusBadge = document.getElementById('statusBadge');
const statusText = document.getElementById('statusText');
const userNameEl = document.getElementById('userName');
const userEmailEl = document.getElementById('userEmail');
const userCompanyEl = document.getElementById('userCompany');
const nextCaptureTimerEl = document.getElementById('nextCaptureTimer');
const sessionTimerEl = document.getElementById('sessionTimer');
const sessionTimerSection = document.getElementById('sessionTimerSection');
const toggleBtn = document.getElementById('toggleBtn');
const loginBtn = document.getElementById('loginBtn');
const logoutBtn = document.getElementById('logoutBtn');
const dashboardBtn = document.getElementById('dashboardBtn');

let timerInterval = null;
let sessionTimerInterval = null;
let logs = [];

// Add log to console
function addLog(message, type = 'info') {
  const timestamp = new Date().toLocaleTimeString();
  const logEntry = `[${timestamp}] ${message}`;
  logs.push(logEntry);
  if (logs.length > 50) logs.shift(); // Keep last 50 logs
  
  const consoleEl = document.getElementById('consoleLog');
  if (consoleEl) {
    consoleEl.innerHTML = logs.map(log => `<div>${log}</div>`).join('');
    consoleEl.scrollTop = consoleEl.scrollHeight;
  }
  
  console.log(logEntry);
}

// Format time in milliseconds to readable format
function formatTime(ms) {
  const seconds = Math.floor(ms / 1000);
  const minutes = Math.floor(seconds / 60);
  const hours = Math.floor(minutes / 60);
  const days = Math.floor(hours / 24);
  
  if (days > 0) {
    return `${days}d ${hours % 24}h ${minutes % 60}m`;
  } else if (hours > 0) {
    return `${hours}h ${minutes % 60}m`;
  } else if (minutes > 0) {
    return `${minutes}m ${seconds % 60}s`;
  } else {
    return `${seconds}s`;
  }
}

// Format session timer with seconds
function formatSessionTime(ms) {
  const seconds = Math.floor(ms / 1000);
  const minutes = Math.floor(seconds / 60);
  const hours = Math.floor(minutes / 60);
  
  const h = hours;
  const m = minutes % 60;
  const s = seconds % 60;
  
  return `${h}h ${m}m ${s}s`;
}

// Convenience wrappers around runtime messages
function sendRuntimeMessage(payload) {
  return new Promise((resolve) => {
    try {
      chrome.runtime.sendMessage(payload, (response) => {
        if (chrome.runtime.lastError) {
          addLog('Runtime message failed: ' + chrome.runtime.lastError.message);
          resolve(null);
        } else {
          resolve(response);
        }
      });
    } catch (error) {
      addLog('Runtime messaging error: ' + error.message);
      resolve(null);
    }
  });
}

async function getExtensionStatus() {
  return sendRuntimeMessage({ action: 'get_status' });
}

async function requestScreenPermissionIfNeeded(hasPermission) {
  if (hasPermission) {
    return { success: true };
  }
  addLog('Requesting screen capture permission before starting...');
  const result = await sendRuntimeMessage({ action: 'request_permission' });
  if (result && result.success) {
    addLog('✅ Screen permission granted');
    showSuccessMessage('Screen permission granted');
    return result;
  }
  addLog('❌ Screen permission denied');
  showSuccessMessage('Permission denied. Cannot start tracking.');
  return { success: false, error: result?.error || 'Permission denied' };
}

// Check authentication status on load
async function checkAuth() {
  addLog('Checking authentication...');
  
  // First check if we have auth in chrome.storage
  let result = await chrome.storage.local.get([
    'auth_token',
    'user_name',
    'user_email',
    'company_id',
    'company_name',
    'company_slug'
  ]);
  
  // If not found, try to sync from page's localStorage via content script
  if (!result.auth_token) {
    addLog('No auth in extension storage, checking page localStorage...');
    try {
      // Query the active tab for palal.app
      const tabs = await chrome.tabs.query({ url: 'https://palal.app/*' });
      if (tabs && tabs.length > 0) {
        // Send message to content script to get localStorage
        const response = await chrome.tabs.sendMessage(tabs[0].id, { action: 'get_local_storage' });
        if (response && response.auth_token) {
          addLog('Found auth in page localStorage, syncing to extension...');
          // Store in chrome.storage
          await chrome.storage.local.set({
            auth_token: response.auth_token,
            user_id: response.user_id,
            user_name: response.user_name,
            user_email: response.user_email,
            company_id: response.company_id,
            company_name: response.company_name,
            company_slug: response.company_slug
          });
          result = response;
        }
      }
    } catch (e) {
      addLog('Could not sync from localStorage: ' + e.message);
    }
  }
  
  if (result.auth_token) {
    addLog('User authenticated: ' + result.user_name);
    // User is logged in
    showTrackerView();
    userNameEl.textContent = result.user_name || 'N/A';
    userEmailEl.textContent = result.user_email || 'N/A';
    userCompanyEl.textContent = formatCompanyContext(result);
    updateStatus();
    // Update status every 2 seconds
    setInterval(updateStatus, 2000);
  } else {
    addLog('User not authenticated');
    // User is not logged in
    showLoginView();
  }
}

function formatCompanyContext(data = {}) {
  const name = data.company_name || 'Unknown workspace';
  const slug = data.company_slug || '';
  if (slug && slug !== 'undefined') {
    return `${name} (${slug})`;
  }
  return name;
}

// Show login view
function showLoginView() {
  loginView.style.display = 'block';
  trackerView.style.display = 'none';
  userCompanyEl.textContent = '--';
}

// Show tracker view
function showTrackerView() {
  loginView.style.display = 'none';
  trackerView.style.display = 'block';
}

// Update UI with current status
async function updateStatus() {
  chrome.runtime.sendMessage({ action: 'get_status' }, async (response) => {
    console.log('Status response:', response);
    if (!response) return;

    // Update tracking status
    if (response.isTracking) {
      statusBadge.classList.add('active');
      statusText.textContent = 'Active';
      toggleBtn.textContent = 'Stop Tracking';
      toggleBtn.classList.add('stop');
      
      // Show and start session timer
      sessionTimerSection.style.display = 'block';
      startSessionTimer(response.sessionStartTime);
      
      // Start timer countdown
      startTimer(response.nextCapture);
    } else {
      statusBadge.classList.remove('active');
      statusText.textContent = 'Inactive';
      toggleBtn.textContent = 'Start Tracking';
      toggleBtn.classList.remove('stop');
      
      // Hide and stop session timer
      sessionTimerSection.style.display = 'none';
      stopSessionTimer();
      
      // Stop timer
      stopTimer();
      nextCaptureTimerEl.textContent = 'Stopped';
    }

    // Update session times
    if (response.sessionStats) {
      if (response.sessionStats.currentSession > 0) {
        addLog('Current session: ' + formatTime(response.sessionStats.currentSession));
      }
    }
  });
}

// Start countdown timer
function startTimer(nextCaptureTime) {
  stopTimer(); // Clear any existing timer
  
  if (!nextCaptureTime) {
    nextCaptureTimerEl.textContent = 'Calculating...';
    return;
  }
  
  timerInterval = setInterval(() => {
    const now = new Date();
    const next = new Date(nextCaptureTime);
    const diff = next - now;
    
    if (diff <= 0) {
      nextCaptureTimerEl.textContent = 'Capturing...';
      updateStatus(); // Refresh to get new time
    } else {
      const seconds = Math.floor(diff / 1000);
      nextCaptureTimerEl.textContent = `${seconds}s`;
    }
  }, 100);
}

// Stop countdown timer
function stopTimer() {
  if (timerInterval) {
    clearInterval(timerInterval);
    timerInterval = null;
  }
}

// Start session timer
function startSessionTimer(sessionStartTime) {
  stopSessionTimer(); // Clear any existing timer
  
  if (!sessionStartTime) {
    sessionTimerEl.textContent = '0h 0m 0s';
    return;
  }
  
  function updateSessionTimer() {
    const now = Date.now();
    const elapsed = now - sessionStartTime;
    sessionTimerEl.textContent = formatSessionTime(elapsed);
  }
  
  // Update immediately
  updateSessionTimer();
  
  // Update every second
  sessionTimerInterval = setInterval(updateSessionTimer, 1000);
}

// Stop session timer
function stopSessionTimer() {
  if (sessionTimerInterval) {
    clearInterval(sessionTimerInterval);
    sessionTimerInterval = null;
  }
  sessionTimerEl.textContent = '0h 0m 0s';
}

// Format relative time
function formatRelativeTime(date) {
  const now = new Date();
  const diffMs = now - date;
  const diffSecs = Math.floor(diffMs / 1000);
  const diffMins = Math.floor(diffSecs / 60);
  const diffHours = Math.floor(diffMins / 60);
  const diffDays = Math.floor(diffHours / 24);

  if (diffMs < 0) {
    // Future time
    const absDiffSecs = Math.abs(diffSecs);
    if (absDiffSecs < 60) {
      return `in ${absDiffSecs}s`;
    }
    const absDiffMins = Math.abs(diffMins);
    return `in ${absDiffMins}m`;
  }

  // Past time
  if (diffSecs < 60) {
    return `${diffSecs}s ago`;
  } else if (diffMins < 60) {
    return `${diffMins}m ago`;
  } else if (diffHours < 24) {
    return `${diffHours}h ago`;
  } else {
    return `${diffDays}d ago`;
  }
}

// Show success message
function showSuccessMessage(message) {
  const existingMsg = document.querySelector('.success-message');
  if (existingMsg) {
    existingMsg.remove();
  }

  const msg = document.createElement('div');
  msg.className = 'success-message';
  msg.textContent = message;
  document.body.appendChild(msg);

  setTimeout(() => {
    msg.remove();
  }, 3000);
}

// Login button - redirect to website
loginBtn.addEventListener('click', () => {
  chrome.tabs.create({ url: 'https://palal.app/login.php' });
});

// Logout button
logoutBtn.addEventListener('click', async () => {
  // Clear auth token
  await chrome.storage.local.remove([
    'auth_token',
    'user_name',
    'user_email',
    'user_id',
    'company_id',
    'company_name',
    'company_slug'
  ]);
  
  // Stop tracking
  chrome.runtime.sendMessage({ action: 'stop_tracking' });
  
  // Show login view
  showLoginView();
  userCompanyEl.textContent = '--';
  showSuccessMessage('Logged out successfully');
});

// Dashboard button
dashboardBtn.addEventListener('click', () => {
  chrome.tabs.create({ url: 'https://palal.app/dashboard.php' });
});

// Toggle tracking (now auto-requests permission when needed)
toggleBtn.addEventListener('click', async () => {
  const status = await getExtensionStatus();
  if (!status) return;

  if (status.isTracking) {
    const stopResult = await sendRuntimeMessage({ action: 'stop_tracking' });
    if (stopResult?.success) {
      showSuccessMessage('Tracking stopped');
      updateStatus();
    }
    return;
  }

  const permissionResult = await requestScreenPermissionIfNeeded(status.hasPermission);
  if (!permissionResult.success) {
    return;
  }

  const manualStartToken = `manual-${Date.now()}-${Math.random().toString(36).slice(2, 8)}`;
  await chrome.storage.local.set({
    manual_start_token: manualStartToken,
    manual_start_requested_at: Date.now()
  });

  const startResult = await sendRuntimeMessage({
    action: 'start_tracking',
    manualStartToken
  });
  if (startResult?.success) {
    showSuccessMessage('Tracking started');
    updateStatus();
  } else if (startResult) {
    showSuccessMessage(startResult.message || 'Unable to start tracking');
  }
});

// Listen for messages from auth.html (after login) and background logs
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'login_success') {
    checkAuth();
  } else if (request.action === 'log') {
    // Receive logs from background script
    addLog(request.message);
  }
});

// Initialize
checkAuth();
