"""
Icon Generator for Employee Tracker Extension

This script generates simple icon files for the Chrome extension.
Requires: pip install pillow
"""

from PIL import Image, ImageDraw
import os

def create_icon(size, filename):
    """Create a simple camera icon"""
    # Create image with blue background
    img = Image.new('RGB', (size, size), color='#2196F3')
    draw = ImageDraw.Draw(img)
    
    # Calculate proportions
    margin = size // 5
    body_width = size - (margin * 2)
    body_height = int(body_width * 0.7)
    body_x = margin
    body_y = margin + size // 10
    
    # Draw camera body (white rectangle)
    draw.rectangle(
        [body_x, body_y, body_x + body_width, body_y + body_height],
        fill='white'
    )
    
    # Draw lens (dark blue circle)
    lens_radius = size // 6
    lens_center_x = size // 2
    lens_center_y = body_y + body_height // 2
    draw.ellipse(
        [
            lens_center_x - lens_radius,
            lens_center_y - lens_radius,
            lens_center_x + lens_radius,
            lens_center_y + lens_radius
        ],
        fill='#1976D2'
    )
    
    # Draw flash (small white rectangle)
    flash_size = max(2, size // 12)
    flash_x = body_x + body_width - flash_size * 2
    flash_y = body_y - flash_size
    draw.rectangle(
        [flash_x, flash_y, flash_x + flash_size, flash_y + flash_size],
        fill='white'
    )
    
    # Save image
    img.save(filename)
    print(f"Created {filename}")

def main():
    # Create icons directory if it doesn't exist
    icons_dir = 'icons'
    if not os.path.exists(icons_dir):
        os.makedirs(icons_dir)
    
    # Generate icons
    create_icon(16, os.path.join(icons_dir, 'icon16.png'))
    create_icon(48, os.path.join(icons_dir, 'icon48.png'))
    create_icon(128, os.path.join(icons_dir, 'icon128.png'))
    
    print("\nAll icons generated successfully!")
    print("Icons are saved in the 'icons' folder")

if __name__ == '__main__':
    try:
        main()
    except ImportError:
        print("Error: PIL (Pillow) is not installed")
        print("Install it with: pip install pillow")
    except Exception as e:
        print(f"Error: {e}")
